﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace AutoMapperPOC.Utils
{
    /// <summary>
    /// Utilities for working with DataTable objects
    /// </summary>
    public class DataTableUtils
    {
        /// <summary>
        /// common use string to hold error conditions that can be returned on exception
        /// if there aren't that many exception scenarios, remove this and just bind message in the actual throw statement
        /// currently only checking for matching column / row value counts ( ie: number of commas ), but invalid chars and encoding might be included later.
        /// </summary>
        private static string _csvFileValidationErrorCondition = "";
        private static readonly string _linebreak = "-------------------------------";

        /// <summary>
        /// Open CSV file from specified path, check that row data value counts match header/col counts, then read into a DataTable and return 
        /// </summary>
        /// <param name="filepath">absolute path to CSV</param>
        /// <returns>DataTable created from CSV</returns>
        public static DataTable ParseFileToDataTable(string filepath)
        {
            if (IsFileDataConsistent(filepath))
            {
                string[] lines = System.IO.File.ReadAllLines(filepath);
                string[] headers = lines[0].Split(',');
                // create storage object
                DataTable dt = new DataTable("csvfile");
                // add headers from file
                foreach (string header in headers)
                {
                    dt.Columns.Add(header);
                }
                // iterate data rows, creating a DataTable row for each and adding values
                for (int i = 1; i < lines.Length; i++)  // start at 2nd row in file ( or DATA row 1 )
                {
                    dt.Rows.Add(lines[i].Split(','));
                }
                return dt;
            }
            else
            {
                throw new ApplicationException($"File at [{filepath}] failed ValidationCheck(s)\n{_csvFileValidationErrorCondition}");
            }
        }

        /// <summary>
        /// Retrieve DataTable ROW1 values and assign to a POCO where its Properties match column names
        /// DataTable is used for this routine due to the ability to specify column DataTypes
        /// TODO: test Date/ DateTime and add supporting util methods to help with conversion
        /// </summary>
        /// <param name="datatable">datatable to read from</param>
        /// <param name="poco">an object with Properties</param>
        /// <returns>object with Properies assigned from row with matching DataTable columns</returns>
        public static object FetchDataTableToPOCO(DataTable datatable, object poco)
        {
            DataTable dtcopy = SetDataTableColumnTypesByPOCOPropertyType(datatable, poco);

            /** import 1st row of data from passed datatable
             * this SHOULD copy the string data and autobox it to destination DataType
             * DateTime might pose a problem here
             * TODO: test Date/ DateTime and add supporting util methods to help with conversion
             */
            dtcopy.ImportRow(datatable.Rows[0]);
            return AssignDataTableValuesToPOCOProperties_1Row(dtcopy, poco);
        }

        public static List<object> FetchDataTableToPOCO_MultiRow(DataTable datatable, object poco)
        {
            DataTable dtcopy = SetDataTableColumnTypesByPOCOPropertyType(datatable, poco);

            /** import 1st row of data from passed datatable
             * this SHOULD copy the string data and autobox it to destination DataType
             * DateTime might pose a problem here
             * TODO: test Date/ DateTime and add supporting util methods to help with conversion
             */
            foreach(DataRow row in datatable.Rows)
            {
                dtcopy.ImportRow(row);
            }
            return AssignDataTableValuesToPOCOProperties_MultiRow(dtcopy, poco);
        }


        #region PRIVATE UTIL METHODS: DATATABLE TO POCO

        /// <summary>
        /// Parse a POCO's Properties and update DataType for any DataTable columns with matching names.
        /// This is to prepare a DataTable for importing a DataRow that can then be assigned to a POCO properties
        /// </summary>
        /// <param name="datatable">datetable to read from</param>
        /// <param name="poco">object with properties to assign</param>
        /// <returns>datatable with updated Column Types where found in the POCO</returns>
        private static DataTable SetDataTableColumnTypesByPOCOPropertyType(DataTable datatable, object poco)
        {
            // copy datatable schema only - no data rows copied
            DataTable dtcopy = datatable.Clone();

            // set data types for any columns matching POCO property names
            foreach (PropertyInfo info in poco.GetType().GetProperties())
            {
                string colName = info.Name;
                Type colType = info.PropertyType;
                // if there's a matching column, set its Type to match the corresponding POCO property type
                var colFound = dtcopy.Columns[colName];
                if (colFound != null)
                {
                    dtcopy.Columns[colName].DataType = colType;
                }
            }
            return dtcopy;

        }


        /// <summary>
        /// Assign values from data table for matching properties
        /// NOTE: DATA Types MUST MATCH or be compatible for AutoBoxing
        /// 
        /// TODO: test Date/ DateTime and add supporting util methods to help with conversion
        /// 
        /// </summary>
        /// <param name="datatable">datetable to read from</param>
        /// <param name="poco">object with properties to assign</param>
        /// <returns>POCO with update Property values where found</returns>
        private static object AssignDataTableValuesToPOCOProperties_1Row(DataTable datatable, object poco, int rowId = 0)
        {
            foreach (PropertyInfo info in poco.GetType().GetProperties())
            {
                // get property name
                string colName = info.Name;
                // search for column name matching property name
                var colFound = datatable.Columns[colName];
                // if a match is found, update its type and 
                if (colFound != null)
                {
                    //Type type = info.GetType();
                    DataRow row = datatable.Rows[rowId];
                    info.SetValue(poco, row[datatable.Columns[colName]]);
                }
            }
            return poco;
        }

        /// <summary>
        /// Assign values for a List of POCOs from data table for matching properties
        /// NOTE: DATA Types MUST MATCH or be compatible for AutoBoxing
        /// 
        /// TODO: test Date/ DateTime and add supporting util methods to help with conversion
        /// 
        /// </summary>
        /// <param name="datatable">datetable to read from</param>
        /// <param name="poco">object with properties to assign</param>
        /// <returns>List of POCO with update Property values where found</returns>
        private static List<object> AssignDataTableValuesToPOCOProperties_MultiRow(DataTable datatable, object poco)
        {
            List<object> pocos = new List<object>();

            for(int rowId=0; rowId<datatable.Rows.Count; rowId++)
            {
                object newPoco = poco.GetType().GetConstructor(Type.EmptyTypes).Invoke(null);
                pocos.Add(AssignDataTableValuesToPOCOProperties_1Row(datatable, newPoco, rowId));
            }
            return pocos;
        }

        #endregion

        #region PRIVATE UTIL METHODS: CSV TO DATATABLE

        /// <summary>
        /// Util method to check that a file exists and throw exception if not
        /// </summary>
        /// <param name="filepath">path to CSV file</param>
        /// <returns>true if file found</returns>
        private static Boolean IsFileExists(string filepath)
        {
            if (!System.IO.File.Exists(filepath))
            {
                throw new System.IO.FileNotFoundException($"FILE PATH:[{filepath}] not found");
            }
            else return true;
        }

        /// <summary>
        /// Util method to verify each data row has same number of values as headers/columns
        /// </summary>
        /// <param name="filepath">absolute path to file</param>
        /// <returns>true if found and passes consistency check(s)</returns>
        private static bool IsFileDataConsistent(string filepath)
        {
            if (IsFileExists(filepath))
            {
                string[] lines = System.IO.File.ReadAllLines(filepath);
                string[] headers = lines[0].Split(',');
                int headerCount = headers.Length;
                bool isValid = true; // set to false if error condition arises

                // get ALL row value counts, trip flag if all rows don't match header count
                for (int row = 1; row < lines.Length; row++)
                {
                    // fetch values in the row
                    string[] values = lines[row].Split(',');
                    // check values in row with expected from the headerCount
                    if (values.Length != headerCount)
                    {
                        _csvFileValidationErrorCondition += $"Number of values in row does not match number of headers:\n{lines[row]}";
                        isValid = false; // trip isValid flag
                    }
                }
                return isValid;
            }
            else
            {
                return false;
            }
        }

        #endregion

        #region PUBLIC UTIL METHODS
        /// <summary>
        /// Util method that reads a DataTable contents to a StringBuilder - normally for output
        /// </summary>
        /// <param name="dt">datatable to read</param>
        /// <returns>StringBuilder of the datatable's contents</returns>
        public static StringBuilder FetchDataTableContents(DataTable dt)
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine("ROW COUNT" + dt.Rows.Count);

            string rowtext = "";
            int idx = 0;

            // column headers
            rowtext += $"COL|";
            foreach (DataColumn col in dt.Columns)
            {
                rowtext += $"{col.ColumnName}|";
            }
            sb.AppendLine(rowtext);

            // row data
            foreach (DataRow row in dt.Rows)
            {
                idx++;
                rowtext = $"#{idx}:|";
                foreach (string value in row.ItemArray)
                {
                    rowtext += $"{value}|";
                }
                sb.AppendLine(rowtext);
            }
            return sb;
        }

        /// <summary>
        /// Util method for outputing DataTable and POCO Property values of non-string types
        /// </summary>
        /// <param name="value">a DataTable cell or POCO Property value</param>
        /// <returns>string representation of value</returns>
        public static string TranslateStringToTypes(object value)
        {
            string valueType = value.GetType().Name;
            switch (valueType)
            {
                case "Int32":
                    return ((Int32)value).ToString();
                case "Double":
                    return ((double)value).ToString();
                case "String":
                    return (String)value;
                default:
                    return null;
            }
        }

        /// <summary>
        /// iterate a POCO and output its Property values
        /// </summary>
        /// <param name="poco"></param>
        public static void OutputPOCOPropValues(object poco)
        {
            Debug.WriteLine($"{_linebreak}");
            Debug.WriteLine($"OUTPUT for object: {poco.GetType().Name}...");
            foreach (PropertyInfo info in poco.GetType().GetProperties())
            {
                string name = info.Name;
                var value = info.GetValue(poco);
                if (value == null)
                {
                    value = "NULL";
                }
                Debug.WriteLine($"PROPERTY:[{name}]: TYPE:[{info.PropertyType.Name}] VALUE:[{DataTableUtils.TranslateStringToTypes(value)}]");
            }
        }

        #endregion
    }
}
