﻿using AutoMapper;
using AutoMapperPOC.Tests.Objects;
using AutoMapperPOC.Utils;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace AutoMapperPOC.Tests.Tests
{
    [TestClass]
    public class AutomapPoco1Tests
    {
        #region VARS
        private string _filePath = @"V:\source\repos\AutoMapperPOC\AutoMapperPOC.Tests\TestData\filedata.csv";
        private string _filePathInvalid = @"ZZ:\source\repos\AutoMapperPOC\AutoMapperPOC.Tests\TestData\filedata.csv";
        string message = "";
        string linebreak = "-------------------------------";

        PocoSource1 ps1 = new PocoSource1()
        {
            Account = "account",
            Name = "name",
            Id = 123,
            Date = "01/01/2023",
            Par = 10001.02
        };
        #endregion

        #region Tests
        [TestMethod]
        [Description("Automapper from one class to antoher - working")]
        public void AutomapPoco1Test()
        {
            var config = new MapperConfiguration(cfg => cfg.CreateMap<PocoSource1, PocoDest1>());
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(ps1);
            // output resulting POCO prop assignments
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }

        [TestMethod]
        [Description("Working functionality to read from CSV and assign POCO properties from any passed Class")]
        public void CSVToTPocoTest()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);
            Debug.WriteLine(DataTableUtils.FetchDataTableContents(dt).ToString());
            var result = DataTableUtils.FetchDataTableToPOCO(dt, new PocoDest1());
            PocoDest1 pd1 = (PocoDest1)result;
            // output resulting POCO prop assignments
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }

        [TestMethod]
        [Description("EndToEnd Test: Read CSV to POCO, Use Automapper to copy from from one POCO to antoher - working")]
        public void CsvToMultiPocoTest()
        {
            // assign POCOSOURCE values from CSV
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);
            Debug.WriteLine(DataTableUtils.FetchDataTableContents(dt).ToString());
            List<object> pocos = DataTableUtils.FetchDataTableToPOCO_MultiRow(dt, new PocoSource1());
            int idx = 0;
            foreach (PocoSource1 poco in pocos)
            {
                Debug.WriteLine($"================================\n#{++idx}");
                // output resulting POCO prop assignments
                DataTableUtils.OutputPOCOPropValues(poco);
                CheckObjectPropertiesForNullValues(poco);
            }
        }

        [TestMethod]
        [Description("EndToEnd Test: Read CSV to POCO, Use Automapper to copy from from one POCO to antoher - working")]
        public void CsvToMultiPocoToAutomapPocoDest1Test()
        {
            // assign POCOSOURCE values from CSV
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);
            Debug.WriteLine(DataTableUtils.FetchDataTableContents(dt).ToString());
            List<object> pocos = DataTableUtils.FetchDataTableToPOCO_MultiRow(dt, new PocoSource1());
            int idx = 0;
            foreach (PocoSource1 poco in pocos)
            {
                Debug.WriteLine($"DT>POCO ================================\n#{++idx}");
                // output resulting POCO prop assignments
                DataTableUtils.OutputPOCOPropValues(poco);
                CheckObjectPropertiesForNullValues(poco);
            }

            idx = 0;
            foreach (PocoSource1 poco in pocos)
            {
                Debug.WriteLine($"SRC>MAP>DEST ================================\n#{++idx}");

                // automap from POCOSOURCE to POCODEST
                var config = new MapperConfiguration(cfg => cfg.CreateMap<PocoSource1, PocoDest1>());
                var mapper = new Mapper(config);
                PocoDest1 pd1 = mapper.Map<PocoDest1>(poco);
                // output resulting POCO prop assignments
                DataTableUtils.OutputPOCOPropValues(pd1);
                CheckObjectPropertiesForNullValues(pd1);
            }
        }

        [TestMethod]
        [Description("Automapper POC attempt from DataTable WITH TYPED COLUMNS to POCO - FAIL")]
        public void AutomapCSVToPoco1Test()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);

            // TYPED headers with single row of data
            DataTable dtclone = dt.Clone();
            foreach(PropertyInfo info in ps1.GetType().GetProperties())
            {
                string colName = info.Name;
                Type colType = info.PropertyType;
                // if there's a matching column, set its Type to match the corresponding POCO property type
                var colFound = dtclone.Columns[colName];
                if(colFound != null)
                {
                    dtclone.Columns[colName].DataType = colType;
                }
            }

            DataRow row = dt.Rows[0];
            dtclone.ImportRow(row); // adds row and preserves DataType of columns

            var config = new MapperConfiguration(cfg => cfg.CreateMap<DataTable, PocoDest1>());
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(dtclone);
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }



        [TestMethod]
        [Description("This doesn't throw exception, but AutoMapper needs CustomResolver or Profile to work with DataTable/DataRow")]
        public void AutomapCSVToPoco2Test()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);
            DataRow row = dt.Rows[0];
            
            var config = new MapperConfiguration(cfg => cfg.CreateMap<DataRow, PocoDest1>());
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(row);

            // no assignments made by Automapperf
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }


        [TestMethod]
        [Description("POC Test: Automapper with DataTable")]
        public void CSVToPocoDataTableTest()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);

            // headers with single row of data
            DataTable dtclone = dt.Clone();
            dtclone.ImportRow(dt.Rows[0]);// row1

            //////////////////////////////////////////////////////
            // This will NOT read from datatable and assign to POCO props => needs CustomResolver
            var config = new MapperConfiguration(cfg => cfg.CreateMap<DataTable, PocoDest1>());
            //////////////////////////////////////////////////////
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(dtclone);

            // output resulting POCO prop assignments
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }

        [TestMethod]
        [Description("POC Test: Automapper with DataRow")]
        public void CSVToPocoDataRowTest()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);

            // headers with single row of data
            DataTable dtclone = dt.Clone();
            dtclone.ImportRow(dt.Rows[0]);// row1

            //////////////////////////////////////////////////////
            // This will NOT read from datatable and assign to POCO props => needs CustomResolver
            var config = new MapperConfiguration(cfg => cfg.CreateMap<DataRow, PocoDest1>());
            //////////////////////////////////////////////////////
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(dtclone.Rows[0]);

            // output resulting POCO prop assignments
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }

        [TestMethod]
        [Description("POC Test: Automapper with DataTable-Duplicate?")]
        public void CSVToPocoDataTable2Test()
        {
            DataTable dt = DataTableUtils.ParseFileToDataTable(_filePath);

            // TYPED headers with single row of data
            DataTable dtclone = dt.Clone();
            foreach (PropertyInfo info in ps1.GetType().GetProperties())
            {
                string colName = info.Name;
                Type colType = info.PropertyType;
                // if there's a matching column, set its Type to match the corresponding POCO property type
                var colFound = dtclone.Columns[colName];
                if (colFound != null)
                {
                    dtclone.Columns[colName].DataType = colType;
                }
            }
            DataRow row = dt.Rows[0];
            dtclone.ImportRow(row); // adds row and preserves DataType of columns

            var config = new MapperConfiguration(cfg => cfg.CreateMap<DataTable, PocoDest1>());
            var mapper = new Mapper(config);
            PocoDest1 pd1 = mapper.Map<PocoDest1>(dtclone);

            // output resulting POCO prop assignments
            DataTableUtils.OutputPOCOPropValues(pd1);
            CheckObjectPropertiesForNullValues(pd1);
        }

        #endregion

        #region UTIL METHODS

        /// <summary>
        /// iterate an object's Properties and check for null
        /// numberic properties will likely be set to 0
        /// </summary>
        /// <param name="poco"></param>
        private void CheckObjectPropertiesForNullValues(object poco)
        {
            Debug.WriteLine($"{linebreak}");
            Debug.WriteLine($"NULL CHECK for object: {poco.GetType().Name}...");
            foreach (PropertyInfo info in poco.GetType().GetProperties())
            {
                string name = info.Name;
                var value = info.GetValue(poco);
                Debug.WriteLine($"PROPERTY:[{name}] VALUE:[{DataTableUtils.TranslateStringToTypes(value)}]");
                string assertionMessage = "Property Value is NULL";
                if (value.GetType() == typeof(string))
                {
                    Assert.IsNotNull(value, assertionMessage);
                }
                else if (value.GetType() == typeof(Int32))
                {
                    Assert.IsTrue((Int32)value != 0, assertionMessage);
                }
                else if (value.GetType() == typeof(double))
                {
                    Assert.IsTrue((double)value != 0, assertionMessage);
                }
                else
                {
                    Assert.Fail("DataType not supported by this test");
                }
            }
        }

        #endregion


        #region AutoMapper Notes for DataTable CustomResolver - not valid for AutoMapper 10.x
        /// <summary>
        /// Test Code for Automapper & generics - not supported by AM 10.1
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="dt"></param>
        /// <returns></returns>
        private List<T> ReadData<T>(DataTable dt)
        {
            //return Mapper.DynamicMap<IDataReader>, List < T >> (dt.CreateDataReader());

            // usage
            DataTable data = new DataTable();
            List<PocoDest1> peopleList = ReadData<PocoDest1>(data);
            return null;
        }
        #endregion
    }
}
